import { CopyFactoryStrategyIdAndName } from './history.client';
import DomainClient from '../domain.client';

/**
 * CopyFactory client for strategy signal requests
 */
export default class StrategySignalClient {

  /**
   * Constructs CopyFactory strategy signal client instance
   * @param {string} accountId strategy provider id
   * @param {string} strategyId strategy id
   * @param {Object} host host data
   * @param {DomainClient} domainClient domain client
   */
  constructor(accountId: string, strategyId: string, host: Object, domainClient: DomainClient);

  /**
   * Generates random signal id
   * @return {String} signal id
   */
  generateSignalId(): string;

  /**
   * Returns active external signals of a strategy. Requires access to
   * copyfactory-api:rest:public:external-signals:getSignals method which is included into reader role.
   * Requires access to strategy, account resources.
   * @returns {Promise<Array<CopyFactoryExternalSignal>>}
   */
  getExternalSignals(): Promise<Array<CopyFactoryExternalSignal>>

  /**
   * Updates external signal for a strategy. See
   * https://metaapi.cloud/docs/copyfactory/restApi/api/trading/updateExternalSignal/
   * @param {String} signalId external signal id (should be 8 alphanumerical symbols)
   * @param {CopyFactoryExternalSignalUpdate} signal signal update payload
   * @return {Promise} promise which resolves when the external signal is updated
   */
  updateExternalSignal(signalId: string, signal: CopyFactoryExternalSignalUpdate): Promise<any>;

  /**
   * Updates external signal for a strategy. See
   * https://metaapi.cloud/docs/copyfactory/restApi/api/trading/removeExternalSignal/
   * @param {String} signalId external signal id
   * @param {CopyFactoryExternalSignalRemove} signal signal removal payload
   * @return {Promise} promise which resolves when the external signal is removed
   */
  removeExternalSignal(signalId: string, signal: CopyFactoryExternalSignalRemove): Promise<any>;

}

/**
 * CopyFactory external signal update payload
 */
export declare type CopyFactoryExternalSignalUpdate = {

  /**
   * trade symbol
   */
  symbol: string,

  /**
   * trade type (one of POSITION_TYPE_BUY, POSITION_TYPE_SELL, ORDER_TYPE_BUY_LIMIT, ORDER_TYPE_SELL_LIMIT, 
   * ORDER_TYPE_BUY_STOP, ORDER_TYPE_SELL_STOP)
   */
  type: string,

  /**
   * time the signal was emitted at
   */
  time: Date,

  /**
   * last time of the signal update
   */
  updateTime?: Date,

  /**
   * volume traded
   */
  volume: number,
  
  /**
   * expert advisor id
   */
  magic?: number,

  /**
   * stop loss price
   */
  stopLoss?: number,

  /**
   * take profit price
   */
  takeProfit?: number,

  /**
   * pending order open price
   */
  openPrice?: number
}

/**
 * CopyFactory external signal
 */
export declare type CopyFactoryExternalSignal = CopyFactoryExternalSignalUpdate & {

  /**
   * external signal id
   */
  id: string

}

/**
 * CopyFactory external signal remove payload
 */
export declare type CopyFactoryExternalSignalRemove = {

  /**
   * the time signal was removed (closed) at
   */
  time: Date
}
